#!/bin/bash
# Modified for usage with eTRIKS deployments
# -*- https://www.etriks.org -*-
# Original version available from https://mran.microsoft.com/install/mro/3.2.5/RevoMath-3.2.5.tar.gz

# For improved Ansible usage (mostly enabling idempotence), return values are modified as the following:
# - '0' is returned when the MKL installation was already done properly, i.e. nothing was to be done.
# - '1' is returned when any error occurs during the installation process.
# - '2' is returned when changes are made (i.e. MKL installed).

# Global variables
# ----------------
PWDD=`pwd`              # Current working directory
RRO_VERSION_MAJOR=3     # R distribution major version number
RRO_VERSION_MINOR=2.5   # R distribution minir version numbers
R_PATH="noop"           # R path
LIB_PATH=               # R /lib path
ETC_PATH=               # R /etc path
LIBRARY_PATH=           # R /library path
# Log variables
LOGDIR='.'              # Log directory path
LOGFILE='mkl.log'       # Log file name
# Run variables
reinstall=$false        # Whether the MKL libraries should be reinstalled

# Log Helper
# ----------
# Params : message to log
log(){
  local message=$1

  # Checking if message is empty
  if [ -z "$message" ] ; then
    echo "Empty log message !" 1>&2
    exit 1
  fi

  # Logging the message as expected
  echo "$(date +%Y/%m/%d-%T) $message" 2>&1 | tee -a ${LOGDIR}/${LOGFILE}
}


# Run the requested action
# ------------------------
# Params: action to run (either 'install' or 'remove')
doAction() {
  log ">>> Running action: $1 <<<"
  if [ "$1" = "install" ] ; then
    installMklLibraries
    break
  elif [ "$1" = "reinstall" ] ; then
    reinstall=$true
    doAction install
  elif [ "$1" = "remove" ] ; then
    uninstallMklLibraries
    break
  else
    log "Unrecognized action $1"
    log "* Should be one of the following: 'install', 'remove'."
    log "* Exiting now........"
    exit 1
  fi
}


# Check the R distribution installation
# -------------------------------------
checkForValidRROInstallation() {
  log "Checking the R distribution installation"

  log "* Checking at the default installation path"
  if [ -e /usr/lib64/MRO-3.2.5/R-3.2.5/lib64/R/bin/R ]; then
    log "* -> Found R at the default installation path!"
    R_PATH=/usr/lib64/MRO-3.2.5/R-3.2.5/lib64/R/bin/R
  elif [ -e /usr/bin/R ]; then
    log "* Getting R path from /usr/bin symlink"
    TEMP_PATH=`ls -l /usr/bin/R`
    R_PATH=`echo $TEMP_PATH | awk -F "[ >]" '{print $NF}'`
    if [ ! -e $R_PATH ]; then
      log "* -> R symlink found in /usr/bin but R can't be found at the designated location."
      log "* Please check your R installation"
      return 1
    else
      log "* -> Found R at $R_PATH"
    fi
  else
    log "* Could not find a valid installation of Microsoft R Open ${RRO_VERSION_MAJOR}.${RRO_VERSION_MINOR}."
    return 1
  fi

  log "Checking the R distribution version"
  RRO=`echo "print(Revo.version)" | $R_PATH -q --no-save`
  MAJOR=`echo $RRO  | awk -F" " '{ for(i=1;i<=NF;i++){ if($i~/major/) {{print $(i+1)} exit }}}'`
  MINOR=`echo $RRO  | awk -F"[ -]" '{ for(i=1;i<=NF;i++){ if($i~/minor/) {{print $(i+1)} exit }}}'`

  if [ "$MAJOR" = $RRO_VERSION_MAJOR ] && [ "$MINOR" = $RRO_VERSION_MINOR ]; then
    log "* -> The R distribution is the correct version."
    log "* Getting library paths"
    PATH_LEN=${#R_PATH}
    LIB_PATH_LEN=`expr $PATH_LEN - 6`
    LIB_PATH=${R_PATH:0:$LIB_PATH_LEN}
    ETC_PATH=$LIB_PATH/etc
    LIBRARY_PATH=$LIB_PATH/library
    LIB_PATH=$LIB_PATH/lib
    if [ -e $LIB_PATH ]; then
      log "* -> Library paths are valid."
    else
      log "* -> Library paths are invalid."
      return 1
    fi
  else
    log "* Error: Microsoft R Open v${MAJOR}.${MINOR} was detected; however, it is not the correct version."
    log "* This utility is for version ${RRO_VERSION_MAJOR}.${RRO_VERSION_MINOR} only."
    return 1
  fi

  log "Ensuring libRlapack.so and libRblas.so exist"
  if [ -e $LIB_PATH/libRlapack.so ] && [ -e $LIB_PATH/libRblas.so ]; then
    log "* -> R libraries found as expected!"
  else
    log "* Error: This does not look like a valid Microsoft R Open installation."
    log "* Libraries libRlapack.so and/or libRblas.so do not exist in ${LIB_PATH}"
    return 1
  fi

  log "The R distribution installation is OK."
  return 0
}


# Check existence of a previous MKL installation
# ----------------------------------------------
checkPreviousMklInstall() {
  log "Checking for previous MKL installations"
  if [ -e $LIB_PATH/libmkl_core.so ] ; then
    log "* A previous installation of MKL was detected."
    return 0
  else
    log "* No MKL installation found."
    return 1
  fi
}


# Copy the MKL library files
# --------------------------
copyLinkMklLibraries() {
  log "Installing MKL libraries"
  log "* Backing up R librairies"
  mv $LIB_PATH/libRlapack.so $LIB_PATH/libRlapack.so.keep
  mv $LIB_PATH/libRblas.so $LIB_PATH/libRblas.so.keep

  log "* Copying MKL libraries"
  cp $PWDD/mkl/libs/* $LIB_PATH

  log "* Setting env variables in Rprofile.site"
  sed -i -e '1 aSys.setenv("MKL_INTERFACE_LAYER"="GNU,LP64")\nSys.setenv("MKL_THREADING_LAYER"="GNU")' $ETC_PATH/Rprofile.site

  log "* Installing RevoUtilsMath"
  if ! $R_PATH CMD INSTALL $PWDD/RevoUtilsMath.tar.gz -q --vanilla --no-save 2>&1 >> ${LOGDIR}/${LOGFILE} ; then
    log "Error: R failed to comply!"
    log "Exiting now......."
    exit 1
  fi

  log "* Activating RevoUtilsMath in Rprofile.site"
  echo "options(defaultPackages=c(getOption('defaultPackages'), 'RevoUtilsMath'))" >> $ETC_PATH/Rprofile.site
  echo ".First <- function(){ library('RevoUtilsMath'); setMKLthreads()}" >> $ETC_PATH/Rprofile.site

  if [ -e $LIBRARY_PATH/RevoUtilsMath/libs/RevoUtilsMath.so ] ; then
    log "MKL was successfully installed for Microsoft R Open ${RRO_VERSION_MAJOR}.${RRO_VERSION_MINOR}."
    log "Exiting now......."
    return 0
  else
    log "Error: MKL installation failed!"
    log "Exiting now......."
    exit 1
  fi
}


# Remove the MKL libraries
# ------------------------
removeLinkMklLibraries() {
  log "* Removing MKL libraries"
  rm $LIB_PATH/libmkl*
  rm $LIB_PATH/libiomp5.so
  rm $LIB_PATH/libRblas.so
  rm $LIB_PATH/libRlapack.so

  log "* Restoring R libraries"
  mv $LIB_PATH/libRblas.so.keep $LIB_PATH/libRblas.so
  mv $LIB_PATH/libRlapack.so.keep $LIB_PATH/libRlapack.so

  log "* Removing env variables from Rprofile.site"
  sed -i -e '/MKL_INTERFACE_LAYER/d' $ETC_PATH/Rprofile.site
  sed -i -e '/MKL_THREADING_LAYER/d' $ETC_PATH/Rprofile.site

  log "* Uninstalling RevoUtilsMath"
  if ! echo "remove.packages('RevoUtilsMath')" | $R_PATH -q --vanilla --no-save 2>&1 >> ${LOGDIR}/${LOGFILE} ; then
    log "Error: R failed to comply!"
    log "Exiting now......."
    exit 1
  fi

  log "* Deactivating RevoUtilsMath in Rprofile.site"
  sed -i -e "/options(defaultPackages=c(getOption('defaultPackages'), 'RevoUtilsMath'))/d" $ETC_PATH/Rprofile.site
  sed -i -e "/.First <- function(){ library('RevoUtilsMath'); setMKLthreads()}/d" $ETC_PATH/Rprofile.site

  if [ -e $LIBRARY_PATH/RevoUtilsMath/libs/RevoUtilsMath.so ] ; then
    log "Error: Couldn't remove MKL libraries."
    log "Exiting now......."
    exit 1
  else
    log "MKL was successfully uninstalled from $LIB_PATH"
    log "Exiting now......."
    return 0
  fi
}


# Install the MKL libs
# --------------------
installMklLibraries() {
  if checkForValidRROInstallation ; then
    if checkPreviousMklInstall ; then
      if reinstall ; then
        removeLinkMklLibraries
      else
        log "* MKL libraries already installed. Nothing to do."
        log "Exiting now......"
        exit 0
      fi
    fi

    copyLinkMklLibraries
    exit 2  ## We changed something
  else
    log "Exiting now....."
    exit 1
  fi
}


# Uninstall the MKL libs
# ----------------------
uninstallMklLibraries() {
  log "Uninstalling MKL"
  if checkForValidRROInstallation ; then
    if checkPreviousMklInstall ; then
      removeLinkMklLibraries
      exit 2  ## We changed something
    else
      log "MKL was not installed in $LIB_PATH"
      exit 0
    fi
  else
    log "Exiting......"
    exit 1
  fi
}

# Doing the magic
# ---------------
# Running requested action
doAction $1

# THE END...

