###################
## Calculate SFT ##
###################
# set possible power
powers <- c(1:30)

# Calculate sfts values for the dataset and network type
sft <- eventReactive(input$calc_sft,{
  # Get precomputed sft values
  if(input$dataset == "fml"){
    if(input$network_type=="unsigned"){
      tmp <- sft_fem_unsigned
    }else if(input$network_type=="signed"){
      tmp <- sft_fem_signed
    }else if(input$network_type=="signed hybrid"){
      tmp <- sft_fem_signed_hybrid
    }
  }else if(input$dataset == "mml"){
    if(input$network_type=="unsigned"){
      tmp <- sft_male_unsigned
    }else if(input$network_type=="signed"){
      tmp <- sft_male_signed
    }else if(input$network_type=="signed hybrid"){
      tmp <- sft_male_signed_hybrid
    }
  # If own data calculate values => not available yet
  }else{
    longProcessStart(session)
    tryCatch({
      tmp <- pickSoftThreshold(data()$datExpr, powerVector = powers, verbose = 0,networkType=input$network_type)},
      finally=longProcessStop(session)
    )
  }
  return(tmp)
})

#######################
## Choose Soft Power ##
#######################
# Plot the sft figures
output$soft_power_plot <- renderPlot(height=function() {min(session$clientData$output_soft_power_plot_width,800)},
				      width=function() {min(session$clientData$output_soft_power_plot_width,1000)},{
  plot_soft_threshold(sft(),powers)
})

# Download sft table
output$downloadsft <- downloadHandler(
  filename = function(){'sft.csv'},
  content = function(file){
    if(!is.null(sft())){
	write.csv(sft()$fitIndices,file=file, row.names=F)
    }else{return()}
  }
)

# Update default value of soft power
observe({
  tmp <- tryCatch(powers[which.max((-sign(sft()$fitIndices[,3])*sft()$fitIndices[,2])>=0.8)],error=function(e){NULL})
  if(!is.null(tmp)){
      updateNumericInput(session,inputId = 'soft_threshold',value = tmp)
  }
})

#######################
## Calculate Network ##
#######################

## set parameters ##
# Add TOM type choice for unsigned network
output$ui_tomtype <- renderUI({ 
  if(input$network_type=='unsigned'){
    selectInput('tom_type',label = 'Choose TOM Type', 
		choices=c('Unsigned'='unsigned','Signed'='signed'),
		selected='unsigned')     
  }
})

# Set TOM type unsigned otherwise
tom_type <- reactive({
  if(!is.null(input$tom_type)){
    input$tom_type
  }else{
    "unsigned"
  }
})

# Add help description for manual construction describing additional parameters
output$textManual <- renderUI({
  if(input$network_construction_method == 'Manual'){
    includeHTML(path="www/html_text/network_construction/create_network2.html")
  }
})

# Add parameter box for manual construction with additional parameters
output$ui_man_param <- renderUI({
  if (input$network_construction_method == 'Manual'){
    wellPanel(
      HTML('<h3 style="margin-top: 10px">Detect Modules</h3>'),
      # Additional Parameters for Module Detection
      flowLayout(
	numericInput('minModuleSize', 'Min Module Size', value = 30,min = 0), 
	numericInput('cutheight', 'Max Joining Height', value = 0.995,min = 0,max=1,step = 0.001),
	numericInput('deepSplit', 'Depth of Split', value = 2,min = 0,max=4,step = 1)
      ),
      # Compute initial modules button
      flowLayout(
	actionButton('define_modules', 'Define Initial Modules')
      ),
      # Additional parameter for module merge
      flowLayout(
	numericInput('MEDissThres', 'Threshold to Merge Modules', value = 0.25,min = 0,max=1,step = 0.001)
      ),
      # Merge module button
      flowLayout(
	actionButton('merge_modules', 'Merge Close Modules')
      )
    )
  }
})

# Set default values for Automatic construction method
man_param <- reactiveValues()
observe({
  if (is.null(input$minModuleSize)){
    man_param$minModuleSize <- 30
    man_param$MEDissThres <- 0.25
    man_param$deepSplit <- 2
    man_param$cutheight <- 0.995
  }else{
    man_param$minModuleSize <- input$minModuleSize
    man_param$MEDissThres <- input$MEDissThres
    man_param$deepSplit <- input$deepSplit
    man_param$cutheight <- input$cutheight
  }
})

# Keep track of initialisation, merging asked and done 
merge_modules <- reactiveValues(value=0,initial=0,true_merged=0)
# keep track of value in case something goes wrong
module <- reactiveValues(current=NULL,old=NULL,initial=NULL)

# Set everything to 0 for Manual, and 1 for Automatic to allow computation
observeEvent(net(),priority=100, {            
  if(input$network_construction_method=='Automatic'){
    merge_modules$value <- 1
    merge_modules$initial <- 1
    merge_modules$true_merged <- 0 # set to 0 no merge happened yet
    module$current <- NULL
    module$old <- NULL
    module$initial <- NULL
  }else{
    merge_modules$value <- 0
    merge_modules$initial <- 0
    merge_modules$true_merged <- 0
    module$current <- NULL
    module$old <- NULL
    module$initial <- NULL
  }
})

# Increment initial value when define initial module button is pressed, and reset the others
observe(priority = 99,{
  input$define_modules
  if(!is.null(input$define_modules)){
    if(input$define_modules != 0){
      isolate({
	  merge_modules$initial <- merge_modules$initial + 1
	  merge_modules$value <- 0
    merge_modules$true_merged <- 0
    module$current <- NULL
	  module$old <- NULL
    module$initial <- NULL
      })
    }
  }
})

# Increment merge value when merge module button is pressed
observe(priority=98,{
  input$merge_modules
  if(!is.null(input$merge_modules)){
      if(input$merge_modules!=0){
	  isolate({
	      merge_modules$value <- merge_modules$value +1
	  })
      }
  }
})

# use absolute correlation or not
useAbs <- reactive({
  if(input$network_type %in% c("unsigned")){
      TRUE
  }else{
      FALSE
  }
})

## calculate network and modules ##
# calculate network
net <- eventReactive(input$calc_network,{
  if(tryCatch({is.na(input$soft_threshold) | !is.numeric(input$soft_threshold) | input$soft_threshold<1 | input$soft_threshold>30},error=function(e){TRUE})){
    return(NULL)
  }else{
    longProcessStart(session)
    tmp <- tryCatch(create_network(data()$datExpr,power=input$soft_threshold,
		    network_construction_method = input$network_construction_method,
		    network_type = input$network_type,man_param = man_param,
		    tom_type = tom_type()),error=function(e){NULL})
    longProcessStop(session)
    return(tmp)
  }
})

# Get initial modules
initial_mod <- reactive({
  net()
  merge_modules$initial
  isolate({
  tmp <- tryCatch(initial_modules(data()$datExpr,power=input$soft_threshold,
		  network_construction_method = input$network_construction_method,
		  man_param = man_param,net=net()),error=function(e){NULL})
  })
  return(tmp)
})

observe({
  initial_mod()
  isolate({
    module$initial <- initial_mod()
  })
})

# merge modules 
merge <- reactive({
  module$initial
  merge_modules$value
  if(merge_modules$value!=0){
  isolate({
    # keep track of old value in case something goes wrong
    merge_old <- module$old 
    # for the first use initial modules definition
    if(merge_modules$true_merged==0){
      merge_new <- tryCatch({mergeModules(datExpr = data()$datExpr, useAbs = useAbs(),
			    network_construction_method = input$network_construction_method,
			    colors=module$initial$dynamicColors,
			    man_param = man_param)},error=function(e){NULL})
    # else use previous merge iteration definition
    }else if (merge_modules$true_merged>=1){
      merge_new <- tryCatch({mergeModules(value=merge_modules$value, datExpr = data()$datExpr, useAbs =useAbs(),
				input$network_construction_method,
			    colors=module$current$moduleColors,man_param)},error=function(e){NULL})
    }
    if(!is.null(merge_new)){
      # increment true merge if everything went ok
      merge_modules$true_merged <- merge_modules$true_merged+1
      return(merge_new)
    }else{
      # keep old value if something went wrong
      return(merge_old)
    }
  })
  }
})


# update merge values 
observe({
    merge()
    isolate({
	if(!is.null(merge())){
	    module$current <- tryCatch(merge(),error=function(e){NULL})
	    module$old <- tryCatch(merge(),error=function(e){NULL})
	}
    })
})
           
# Calculate dissimilarity between module eigengenes   
MEDiss <- reactive({
  module$initial
  module$current
  isolate(tryCatch({
    # use initial definition if no merged done
    if(merge_modules$true_merged==0){
      ME <- module$initial$MEs$eigengenes 
    # use merge definition if already done
    }else if(merge_modules$true_merged >=1){
      ME <- module$current$MEs
    }
    # calculate correlation between eigengenes
    if(ncol(ME)>1){
      if(useAbs()){
	MEDiss <- 1-abs(cor(ME))
      }else {
	MEDiss <- 1-cor(ME)
      }
    }else{
      MEDiss <- 0
    }
  },error=function(e){NULL}))
  return(MEDiss)
})

# Calculate eigengenes dendrogram
METree <- reactive({tryCatch(hclust(as.dist(MEDiss()), method = "average"),error=function(e){NULL})})

## Plot Figures ##
# plot module eigengene dendrogram
output$METree_plot <- renderPlot(
  height = function() {
    if(tryCatch({dim(as.matrix(MEDiss()))[1]>1},error=function(e){FALSE})){
      min(0.5*session$clientData$output_METree_plot_width,500)
    }else{0}
  },
  width = function() {min(session$clientData$output_METree_plot_width,1000)},
  {
    if(tryCatch({dim(as.matrix(MEDiss()))[1]>1},error=function(e){FALSE})){
      plot(as.dendrogram(METree()),main = "Clustering of module eigengenes", xlab = "", sub = "")
      if (input$network_construction_method=='Manual'){
	if(is.numeric(man_param$MEDissThres)){
	  abline(h=man_param$MEDissThres, col = "red")
	}
      }
    }
  }
)

# plot gene dendrogram with module colors
output$network_dendrogram <- renderPlot(height=function() {min(0.5*session$clientData$output_network_dendrogram_width,500)},
					width=function() {min(session$clientData$output_network_dendrogram_width,1000)},
  {
    module$initial
    net()
    module$current
    man_param$cutheight
    isolate(
      tryCatch({
	if(merge_modules$true_merged>=1){
	  plotDendroAndColors( net()$geneTree, colors = cbind(module$initial$dynamicColors, module$current$moduleColors),
				c("Initial Modules", "Merged Modules"),
				dendroLabels = FALSE, hang = 0.03, 
				addGuide = TRUE, guideHang = 0.05,abHeight=man_param$cutheight)
	}else if (merge_modules$true_merged==0){
	  plotDendroAndColors( net()$geneTree, colors = module$initial$dynamicColors,
				"Initial Modules",
				dendroLabels = FALSE, hang = 0.03, 
				addGuide = TRUE, guideHang = 0.05, abHeight=man_param$cutheight)
	}else{
	  plotDendroAndColors( net()$geneTree,
				dendroLabels = FALSE, hang = 0.03, 
				addGuide = TRUE, guideHang = 0.05, abHeight=man_param$cutheight)
	}
      },error=function(e){NULL})
    )
  }
)
        
# prepare module assignment table        
moduleassignment <- reactive({
  if(merge_modules$true_merged >=1){
    assigned <- tryCatch({data.frame('Features'=names(data()$datExpr),"Initial Module"=module$initial$dynamicColors, "Merged Module"=module$current$moduleColors)}, error=function(e){NULL})
  }else if (merge_modules$true_merged ==0){
    assigned <- tryCatch({data.frame('Features'=names(data()$datExpr),"Initial Module"=module$initial$dynamicColors)}, error=function(e){NULL})
  }
  return(list("assignment"=assigned))
})

# download module assignment
output$downloadmoduleassignment <- downloadHandler(
  filename = function(){'module_assignment.csv'},
  content = function(file){
      write.csv(moduleassignment()$assignment,file=file, row.names=F)
  }
)
