#########################################
## Module Eigengene Trait Relationship ##
#########################################

# calculate trait module eigengene correlations and p-values       
moduleTrait <- reactive({
  merge()
  initial_mod()
  isolate({
    tryCatch({
      if(merge_modules$true_merged>=1){
	if(!is.null(merge())){
	  MEs <- orderMEs(merge()$MEs)
	  color <- merge()$moduleColors
	}else{
	  MEs <- orderMEs(initial_mod()$MEs$eigengenes)
	  color <- initial_mod()$dynamicColors
	  # if merge happened but initial modules are used send alert
	  session$sendCustomMessage(type = 'nomerge', message = list(a = initial_mod()))
	}
      }else{
	MEs <- orderMEs(initial_mod()$MEs$eigengenes)
	color <- initial_mod()$dynamicColors
      }
      moduleTraitCor <- cor(MEs, data()$datTraits, use = "p")
      moduleTraitPvalue <- corPvalueStudent(moduleTraitCor, nrow(data()$datExpr))
      textMatrix <- paste(signif(moduleTraitCor, 2), "\n(",signif(moduleTraitPvalue, 1), ")", sep = "");
      dim(textMatrix) <- dim(moduleTraitCor)
      return(list("moduleTraitCor"=moduleTraitCor,
		  "moduleTraitPvalue"= moduleTraitPvalue,
		  "MEs"=MEs,"textMatrix"=textMatrix,'colors'=color))
    },error=function(e){NULL})
  })
})

# download module trait values
output$downloadmoduletrait <- downloadHandler(
  filename = function(){'module_trait_association.csv'},
  content = function(file){
    tmp <- paste(signif(moduleTrait()$moduleTraitCor, 2), " (",
			signif(moduleTrait()$moduleTraitPvalue, 1), ")", sep = "")
    dim(tmp) <- dim(moduleTrait()$moduleTraitCor)
    row.names(tmp) <- names(moduleTrait()$MEs)
    colnames(tmp) <- names(data()$datTraits)
    write.csv(tmp,file=file, row.names=T)
  }
)
        
# plot heatmap
output$moduleTraitMatrix <- renderPlot(height=function() {min(0.5*session$clientData$output_moduleTraitMatrix_width,500)},
				       width=function() {min(session$clientData$output_moduleTraitMatrix_width,1000)},
  {
    moduleTrait()
    par(mar= c(6, 10, 3, 1))
    isolate(                      
      tryCatch({
	labeledHeatmap(Matrix = moduleTrait()$moduleTraitCor,
		       xLabels = names(data()$datTraits),
		       yLabels = names(moduleTrait()$MEs),
		       ySymbols = names(moduleTrait()$MEs),
		       colorLabels = FALSE,
		       colors = bluered(50),
		       textMatrix = moduleTrait()$textMatrix,
		       setStdMargins = TRUE,
		       cex.text = 0.7,cex.lab=0.7,
		       zlim = c(-1,1),
		       main = paste("Module-trait relationships"),
		       xColorOffset=par("cxy")[1]*2,yColorWidth=0.01)
      })
    )
  }
)    

#######################################
## Module genes - Trait relationship ##
#######################################
# ui to choose module and trait of interest
output$module_trait_ui <- renderUI({
  wellPanel(
    flowLayout(
      selectInput('trait_interest', 'Trait of interest', choices = names(data()$datTraits),multiple = F), 
      selectInput('module_interest', 'Choose Module', choices = substring(names(moduleTrait()$MEs), 3),multiple=F)
    )
  )
})       
            
# calculate gene correlation with trait (GS) and module eigengene (MM)
gsMM <- reactive({
  moduleTrait()
  if(!is.null(input$trait_interest)){
    isolate({
      tmp <- tryCatch({gene_relationship(data()$datTraits, data()$datExpr, moduleTrait()$MEs)},error=function(e){NULL})
      return(tmp)
    })
  }
})
    
# plot sample eigengene value for chosen module
output$plot_GSvsMM3 <- renderPlot(height=function() {min(session$clientData$output_plot_GSvsMM3_width,400)},
				  width=function() {min(session$clientData$output_plot_GSvsMM3_width,1000)},
  {
    gsMM()
    if(!is.null(input$trait_interest)){
      input$module_interest
      isolate({
	tryCatch(
	  barplot(moduleTrait()$MEs[, paste("ME",input$module_interest, sep="")], 
		  col=input$module_interest, main="Module Eigengene Expression by Sample", cex.main=2, names.arg=row.names(data()$datExpr),
		  ylab="eigengene expression",xlab="array sample",las=2,cex.names=0.5),
	  error=function(e){NULL})
      })
    }
})
    
# plot Genes Significance with chosen trait for all modules 
output$plot_GSvsMM1 <- renderPlot(height=function() {min(session$clientData$output_plot_GSvsMM1_width,400)},
				  width=function() {min(session$clientData$output_plot_GSvsMM3_width/2,500)},
  {
    gsMM()
    if(!is.null(input$trait_interest)){
      input$module_interest
      isolate({
	tryCatch(
	  plotModuleSignificance(abs(gsMM()$geneTraitSignificance[,paste("GS.",input$trait_interest,sep="")]),moduleTrait()$colors,boxplot = TRUE,cex.axis=0.8,las=3,main="Gene significance across modules\n"),
	  error=function(e){NULL}
	)
      })
    }
  }
)
 
# plot Genes Significance with chosen trait for all modules 
output$plot_GSvsMM2 <- renderPlot(height=function() {min(session$clientData$output_plot_GSvsMM2_width,400)},
                                      width=function() {min(session$clientData$output_plot_GSvsMM3_width/2,500)},
  {
    gsMM()
    if(!is.null(input$trait_interest)){
      input$module_interest
      isolate({
	tryCatch(
	  verboseScatterplot(x = abs(gsMM()$geneModuleMembership[(moduleTrait()$colors == input$module_interest), match(input$module_interest, 					   gsMM()$modNames)]),
			     y = abs(gsMM()$geneTraitSignificance[(moduleTrait()$colors == input$module_interest),paste("GS.",input$trait_interest,sep="")]),
			     xlab = paste("Module Membership in", input$module_interest, "module"),
			     ylab = paste("Gene significance for", input$trait_interest),
			     main = paste("Module membership vs. gene significance\n"),
			     cex.main = 1.2, cex.lab = 1.2, cex.axis = 1.2, pch=21,
			     col ="black", bg = input$module_interest),
	    error=function(e){NULL}
	)
      })
    }
  }
)

# make table of selected genes in the plot GSvsMM2    
selected_Genes_table <- reactive({
  if(!is.null(input$GSvsMM2_brush)){
    isolate({
      xs <- gsMM()$geneModuleMembership[(moduleTrait()$colors == input$module_interest),match(input$module_interest, gsMM()$modNames)]
      ys <- gsMM()$geneTraitSignificance[(moduleTrait()$colors == input$module_interest),paste("GS.",input$trait_interest,sep="")]
      genes <- which( (abs(xs) <= input$GSvsMM2_brush$xmax) & (abs(xs) >= input$GSvsMM2_brush$xmin) & (abs(ys) <= input$GSvsMM2_brush$ymax) & (abs(ys) >= input$GSvsMM2_brush$ymin))
      
      genes_table <- data.frame('Feature'= names(data()$datExpr)[(moduleTrait()$colors == input$module_interest)][genes],
				'Module'= moduleTrait()$colors[moduleTrait()$colors == input$module_interest][genes],
				'Module Membership'=xs[genes],'Module Membership P-value'=gsMM()$MMPvalue[(moduleTrait()$colors == input$module_interest),match(input$module_interest, gsMM()$modNames)][genes],
				'Correlation with Trait'=ys[genes],'Correlation with Trait P-value'=gsMM()$GSPvalue[(moduleTrait()$colors == input$module_interest),paste("p.GS.",input$trait_interest,sep="")][genes])
    })
    return(genes_table)
  }else{NULL}
})

# render table of selected genes    
output$data_selected <- DT::renderDataTable({
  if(!is.null(selected_Genes_table())){
    DT::datatable(selected_Genes_table(),
		  rownames=F,extensions=list(ColReorder=NULL,ColVis=NULL,TableTools=NULL,FixedColumns=list(leftColumns = 1)), 
		  options = list(dom = 'RCT<"clear">lftip',
				 colVis=list(buttonText="Show/Hide"),
				 tableTools = list(sSwfPath = copySWF('www'), aButtons = list(list(sExtends = 'collection',sButtonText = 'Save',aButtons = c('csv', 'xls')))),
				 scrollX = TRUE,scrollXInner="100%",scrollCollapse = FALSE,autoWidth=FALSE))
  }else{}
})

# download whole table
output$downloadgsmm <- downloadHandler(
  filename = function(){paste0('module_', input$module_interest, '_membership_gene_significance_', input$trait_interest, '.csv')},
  content = function(file){
    tryCatch({
    xs <- gsMM()$geneModuleMembership[(moduleTrait()$colors == input$module_interest),match(input$module_interest, gsMM()$modNames)]
    ys <- gsMM()$geneTraitSignificance[(moduleTrait()$colors == input$module_interest),paste("GS.",input$trait_interest,sep="")]
    tmp <- data.frame(names(data()$datExpr)[(moduleTrait()$colors == input$module_interest)],
		      moduleTrait()$colors[moduleTrait()$colors == input$module_interest],
		      xs,gsMM()$MMPvalue[(moduleTrait()$colors == input$module_interest),match(input$module_interest, gsMM()$modNames)],
		      ys,gsMM()$GSPvalue[(moduleTrait()$colors == input$module_interest),paste("p.GS.",input$trait_interest,sep="")])
    names(tmp) <- c('Feature','Module','Module Membership','Module Membership P-value', paste('Correlation with ', input$trait_interest,sep=""),
		  paste('Correlation with ', input$trait_interest,' P-value',sep=""))
    
    write.csv(tmp,file=file, row.names=F)
  })
  }
)  
 
################
## Show graph ##
################
# panel for graph parameters
output$network_param_ui <- renderUI({
  wellPanel(
    flowLayout(
      selectInput(inputId = 'modules_for_network',label = 'Modules',choices =substring(names(moduleTrait()$MEs), 3) ,multiple = T),
      numericInput('link_threshold', label = 'Link threshold',min = 0, max=1,value = 0.1,step=0.01),
      checkboxInput(inputId = 'remove_unlink',label = 'Remove Unlinked Nodes',value = T)
    ),
    actionButton(inputId = 'plot_graph',label = 'Plot Network')
  ) 
})

# make the subset for the network
network_to_show <- eventReactive(input$plot_graph,{
  tryCatch({
    genes <- which(moduleTrait()$colors %in% input$modules_for_network)
    tmp <- net()$adjacency[genes,genes]
    diag(tmp)<- NA
    tmp2 <- apply(tmp,1,function(x){all(x<input$link_threshold,na.rm = TRUE)})
    if(input$remove_unlink){
      genes <- genes[!tmp2]
    }
    if(length(genes)>0){
      value <- as.vector(as.dist(net()$adjacency[genes,genes]))
      tmp <- (value >= input$link_threshold)
      nodes <- row.names(net()$adjacency)[genes]
      srce <- rep(c(0:(length(genes)-1)),times = c((length(genes)-1):0))
      tgt <- c(0:(length(genes)-1))[unlist(lapply(2:length(genes),function(i){i:length(genes)}))]
      group <- moduleTrait()$colors[genes]
      MisLinks <- data.frame("source"=srce,"target"=tgt,"value"=value)[tmp,]
      MisNodes <- data.frame("name"=nodes,"nodeid"=c(0:(length(genes)-1)),"group"=group)
      session$sendCustomMessage(type = 'show',message = list(value=TRUE,div='#networkPlot'))
      session$sendCustomMessage(type = 'disable',message = list(value=TRUE,div='#downloadshownnetwork',parent=FALSE))
      return(list("MisLinks"=MisLinks,"MisNodes"=MisNodes))
    }else{ 
      session$sendCustomMessage(type = 'nonode',message = list(a=genes))
    }
  },error=function(e){NULL})
  session$sendCustomMessage(type = 'show',message = list(value=FALSE, div='#networkPlot'))
  session$sendCustomMessage(type = 'disable',message = list(value=FALSE, div='#downloadshownnetwork',parent=FALSE))
})

# render Graph
output$networkPlot <-renderForceNetwork({
  input$plot_graph
  isolate({
    tmp <- tryCatch(network_to_show()$MisLinks,error=function(e){NULL});
    tmp2 <- tryCatch({network_to_show()$MisNodes}, error=function(e){NULL});
    net <- tryCatch(forceNetwork(Links = tmp, Nodes = tmp2,opacity = 0.8,
			Source = "source", Target = "target",
			Value = "value", NodeID = "name",
			Group = "group",colourScale = JS(paste("d3.scale.ordinal().range(['",paste(col2hex(unique(tmp2$group)),collapse = "', '"),"'])",sep="")),
			zoom = TRUE,bounded=FALSE,linkDistance = JS("function(d){return (1-d.value) * 100}"),
			linkWidth = JS("function(d) { return d.value*3; }")),error=function(e){NULL})
    return(net)
  })
})

# download shown network
output$downloadshownnetwork <- downloadHandler(
  filename = function(){'network.zip'},
  content = function(file){
    tmp <- network_to_show()$MisLinks
    tmp2 <- network_to_show()$MisNodes
    tmpdir <- tempdir()
    wd <- getwd()
    setwd(tmpdir)
    fs <- c("nodes.csv", "links.csv")
    write.csv(tmp, file = "links.csv",row.names=FALSE)
    write.csv(tmp2, file = "nodes.csv",row.names=FALSE)
    
    zip(zipfile=file, files=fs)
    
    file.remove(fs)
    setwd(wd)
      
  },
  contentType = "application/zip"
)

# download whole network
output$downloadwholenetwork <- downloadHandler(
  filename = function(){'network_all.zip'},
  content = function(file){
    longProcessStart(session)
    tryCatch({
    # make tables
      value <- as.vector(as.dist(net()$adjacency))
      nodes <- row.names(net()$adjacency)
      srce <- rep(nodes,times = c((length(nodes)-1):0))
      tgt <- nodes[unlist(lapply(2:length(nodes),function(i){i:length(nodes)}))]
      group <- moduleTrait()$colors
      MisLinks <- data.frame("source"=srce,"target"=tgt,"value"=value)
      MisNodes <- data.frame("name"=nodes,"group"=group)
      
      tmpdir <- tempdir()
      wd <- getwd()
      setwd(tmpdir)
      fs <- c("nodes_all.csv", "links_all.csv")
      write.csv(MisLinks, file = "links_all.csv",row.names=FALSE)
      write.csv(MisNodes, file = "nodes_all.csv",row.names=FALSE)
      
      zip(zipfile=file, files=fs)
      
      file.remove(fs)
      setwd(wd)
    },finally=longProcessStop(session))
  },
  contentType = "application/zip"
)